/* FILE: mscreen.c                              (D. Tottingham  04/28/91)

This is a collection of C functions that manage the screen for xdetect.
All functions have been written and compiled medium.  The following functions
are included:

s_add_channel ()                add a channel to the home display
s_clear ()                      clear the screen
s_initialize ()                 initialize the screen module
s_initialize_screen ()          initialize the graphics screen
s_display_bar ()                display the bar
s_display_ids ()                display station id, channel, and trigger status
s_display_traces ()             display normalized traces on the screen
s_display_triggers ()           display triggers on the screen
s_reset_screen ()               reset the graphics screen to text
s_remove_channel ()             remove a channel from the home display
s_screen_on ()                  return screen flag
s_select_display ()             select the display mode
s_select_screen ()              select a screen
s_set_channelrange ()           set the channel range in the block display
s_view_help ()                  display the help screen

EXTERNAL FUNCTIONS CALLED:

dm_get_first_buffer ()          get first new demux buffer from demux queue
dm_get_next_buffer ()           get next buffer from demux queue
dsp_get_spectral_channel ()     get the spectral channel
dsp_get_status ()               get dsp status
dt_get_channel_gain ()          get channel gain
dt_get_channel_size ()          get channel block size
dt_get_scan_count ()            get scan count
er_abort ()                     display an error message then quit
o_write_logfile ()              write string to log file stream
st_build_textline ()            build a textline for the mscreen module
t_get_channel_trigger_state ()  get trigger state for given channel

HISTORY:
   none

*/


/*************************************************************************
                            INCLUDE FILES


*************************************************************************/
#include <dos.h>
#include <stdio.h>
#include <malloc.h>
#include <string.h>

#include "mconst.h"
#include "mdemux.h"
#include "mdsp.h"
#include "mdt28xx.h"
#include "merror.h"
#include "mhelp.h"
#include "mlog.h"
#include "mscreen.h"
#include "mstation.h"
#include "mtrigger.h"
#include "mutils.h"
#include "mvideo.h"


/*************************************************************************
                                GLOBALS


*************************************************************************/
PRIVATE S_DISPLAY display[N_DISPLAYS], *current_display;
PRIVATE S_HEADER header, *h = &header;
PRIVATE unsigned int old_size, current_size;
PRIVATE int far * raw_data;
PRIVATE int far * plot_data;
PRIVATE int far * old_plot_data;
PRIVATE unsigned int screen_id;
PRIVATE int display_id;
PRIVATE char textline[MAX_TEXTLINE];
PRIVATE FLAG reset_display;


/*=======================================================================*
 *                            build_textline                             *
 *=======================================================================*/
/* Construct a text string for the specified channel.  The string
   contains the channel number, station id, channel status, latitude,
   longitude, and channel gain.                                          */

PRIVATE
int build_textline (channel, textline)
int channel;
char textline[];
{
   int j;

   j = st_build_textline (channel, textline);
   j += sprintf (textline+j, (t_get_channel_trigger_state(channel)) ? " ON " : "    ");
   j += sprintf (textline+j, "%1d", dt_get_channel_gain(channel));

   return (j);
}

/*=======================================================================*
 *                               clear_bar                               *
 *=======================================================================*/
/* Clear bar from the screen.                                            */

PRIVATE
void clear_bar ()
{
   fill ( BIG_RECT_I1, BIG_RECT_J1, BIG_RECT_I2, BIG_RECT_J2, BCKGND_COLOR);
}

/*=======================================================================*
 *                             clear_display                             *
 *=======================================================================*/
/* Clear ids, triggers, and traces from the screen.                      */

PRIVATE
void clear_display ()
{
   int i, j, k, i_increment, npoints;
   unsigned int channel_offset;

   if (current_display != NULL) {
      i_increment = (int) ((VERTICAL_PIXS) / current_display->orientation);
      i = (TOP_MARGIN) + (i_increment / 2);

      channel_offset = HORIZONTAL_PIXS - old_size;
      j = LEFT_MARGIN + channel_offset;

      for (k = 0; k < current_display->n_channels; k++, i += i_increment,
                                          channel_offset += HORIZONTAL_PIXS) {
         gputs ( i-4, 15, "               ", 15, TEXT_OFF_COLORS);
         if (old_plot_data)
            plot( i, j, &old_plot_data[channel_offset], old_size, PLOT_OFF_COLOR);
      }
      if (plot_data) {
         _ffree (plot_data); plot_data = NULL;
      }
      if (old_plot_data) {
         _ffree (old_plot_data); old_plot_data = NULL;
      }
      if (raw_data) {
         _ffree (raw_data); raw_data = NULL;
      }
   }
   old_size = current_size = 0;
   reset_display = TRUE;
}

/*=======================================================================*
 *                             display_bar                               *
 *=======================================================================*/
/* Draw the channel block position indicator on the screen.  The big
   rectangle all channels.  The small rectangle represents the portion
   of and the location of the block that is currently being viewed.  The
   small rectangle is called the block position indicator.               */

PRIVATE
void display_bar (begin_channel, n_channels, max_channel)
unsigned int begin_channel;
unsigned int n_channels;
unsigned int max_channel;
{
   static unsigned top_limit, bottom_limit, old_limit = FALSE;

   /* Draw the big rectangle */
   rectangle ( BIG_RECT_I1, BIG_RECT_J1, BIG_RECT_I2, BIG_RECT_J2, BIG_RECT_COLOR);

   /* Clear the small rectangle if it exists */
   if (old_limit)
      rectangle ( top_limit, SMALL_RECT_J1, bottom_limit, SMALL_RECT_J2, BCKGND_COLOR);

   /* Compute new small rectangle */
   top_limit= begin_channel * BIG_RECT_WDTH / max_channel + (BIG_RECT_I1 + 1);
   bottom_limit = (begin_channel+n_channels) * BIG_RECT_WDTH / max_channel
                  + (BIG_RECT_I1 + 1);

   /* Draw the little rectangle */
   rectangle (top_limit, SMALL_RECT_J1, bottom_limit, SMALL_RECT_J2, SMALL_RECT_COLOR);
   old_limit = TRUE;
}

/*=======================================================================*
 *                            update_header                              *
 *=======================================================================*/
/* Draw the screen header.                                               */

PRIVATE
void update_header ()
{
   static char hdr_string[90], decimation_str[20], shift_str[5];
   char * left_tline, * right_tline;
   int i, magnification;

   left_tline = u_build_timeline (h->left_time, FALSE);
   right_tline = u_build_timeline (h->right_time, FALSE);

   /* Prepare header string */
   if (h->decimation <= 1)
      sprintf (decimation_str, "Undecimated");
   else
      sprintf (decimation_str, "Decimated %2d", h->decimation);

   /* Prepare shift string */
   for (magnification = 1, i = 0; i < h->magnification;
        magnification <<= 1, i++);
   sprintf (shift_str, "%3dX", magnification);

   sprintf (hdr_string, "%.11s%.4s %-14.12s  %7.2f Sps %-4s  %-14s %14.12s",
                      left_tline, &left_tline[24], &left_tline[11],
                      h->srate, shift_str, decimation_str, &right_tline[11]);

   /* Write header strings to screen */
   gputs ( HEADER_I, 4, hdr_string, 90, HEADER_COLORS);

   free (left_tline);
   free (right_tline);
}

/*=======================================================================*
 *                            display_traces                             *
 *=======================================================================*/
/* Display normalized traces on the screen.                              */

PRIVATE
void display_traces (plot_size)
unsigned int plot_size;
{
   unsigned int i, j, old_j, k, i_increment;
   unsigned int channel_offset, old_offset;
   int far * temp;

   channel_offset = HORIZONTAL_PIXS - plot_size;
   old_offset = HORIZONTAL_PIXS - old_size;

   i_increment = (int) ((VERTICAL_PIXS) / current_display->orientation);
   i = (TOP_MARGIN) + (i_increment / 2);
   j = LEFT_MARGIN + channel_offset;
   old_j = LEFT_MARGIN + old_offset;

   update_header ();

   for (k = 0; k < current_display->n_channels; k++, i += i_increment,
                                       channel_offset += HORIZONTAL_PIXS,
                                       old_offset += HORIZONTAL_PIXS) {
      if (old_plot_data != NULL) {
         if (old_size == plot_size)
            plotmod( i, j, &old_plot_data[channel_offset], &plot_data[channel_offset],
                     plot_size, PLOTMOD_COLORS);
         else {
             plot( i, old_j, &old_plot_data[old_offset], old_size, PLOT_OFF_COLOR);
             plot( i, j, &plot_data[channel_offset], plot_size, PLOT_ON_COLOR);
         }
      } else
         plot( i, j, &plot_data[channel_offset], plot_size, PLOT_ON_COLOR);
   }

   temp = plot_data;
   plot_data = old_plot_data;
   old_plot_data = temp;
   old_size = plot_size;
}

/*=======================================================================*
 *                              s_add_channel                            *
 *=======================================================================*/
/* Add a channel to the home display.                                              */

PUBLIC
FLAG s_add_channel (channel)
unsigned int channel;
{
   if (display[HOME].n_channels < MAX_DISPLAY) {
      display[HOME].channel_list[display[HOME].n_channels] = channel;
      display[HOME].n_channels++;
      display[HOME].orientation++;
      if (screen_id) clear_display();
      return (TRUE);
   } else return (FALSE);
}

/*=======================================================================*
 *                                 s_clear                               *
 *=======================================================================*/
/* Clear the screen.                                                     */

PUBLIC
void s_clear ()
{
   turnon( BCKGND_COLOR);
}

/*=======================================================================*
 *                              s_display_bar                            *
 *=======================================================================*/
/* Display the bar.                                                      */

PUBLIC
void s_display_bar ()
{
   display_bar (display[BLOCK].begin_channel, display[BLOCK].n_channels,
                dt_get_scan_count());
}

/*=======================================================================*
 *                              s_display_ids                            *
 *=======================================================================*/
/* Display station id, channel, and trigger status.                      */

PUBLIC
void s_display_ids ()
{
   unsigned int i, k, l, i_increment;
   unsigned int channel;

   if (screen_id != SCREEN_1) return;

   i_increment = (int) ((VERTICAL_PIXS) / current_display->orientation);
   i = (TOP_MARGIN) + (i_increment / 2);

   for (k = 0; k < current_display->n_channels; k++) {
      channel = (current_display->type == LIST) ?
                 current_display->channel_list[k] :
                 current_display->begin_channel + k;
      l = build_textline (channel, textline);
      gputs ( i-4, 15, textline, l+1, TEXT_COLORS);
      i += i_increment;
   }
}

/*=======================================================================*
 *                            s_display_traces                           *
 *=======================================================================*/
/* Display normalized traces on the screen.                              */

PUBLIC
void s_display_traces (rel_decimation, rel_bitshift)
int rel_decimation;
int rel_bitshift;
{
   Q_BUFFER far * b_head;
   double last_time;
   unsigned long channel_blocksize;
   unsigned int i, k, plot_size, channel_offset, index;
   unsigned int initial_offset, copy_size, copy_from, copy_to;
   int far * from_ptr, far * to_ptr, new_bitshift, lsb_mask, adjust;
   FLAG same_buffer;
   unsigned int bit_shift;

   /* Make sure that SCREEN_1 is on */
   if (screen_id != SCREEN_1) return;

   /* Check the demux and display queues */
   if ((b_head = dm_get_first_buffer()) == NULL) return;

   /* Compute decimation and plot_size */
   channel_blocksize = dt_get_channel_size ();
   h->decimation = rel_decimation + ( channel_blocksize <= HORIZONTAL_PIXS) ? 1
                   : channel_blocksize / HORIZONTAL_PIXS;
   if (!h->decimation) h->decimation = 1;
   plot_size = channel_blocksize / h->decimation;

   /* Compute new bit_shift */
   new_bitshift = h->magnification + rel_bitshift;
   if (new_bitshift >= 0 && new_bitshift <= MAX_MAGNIFICATION)
      h->magnification = new_bitshift;

   /* Compute time */
   h->srate = b_head->info.dig_rate;
   last_time = h->right_time;
   h->right_time = b_head->info.begintime + (((double) channel_blocksize) / h->srate);
   h->left_time = h->right_time - (((double) HORIZONTAL_PIXS) * h->decimation / h->srate);
   same_buffer = (h->right_time - last_time) ? FALSE : TRUE;

   if (raw_data == NULL ) {
/*
      raw_data = (int far *) _fmalloc (HORIZONTAL_PIXS * current_display->n_channels * sizeof(int));
*/
      raw_data = (int far *) _fmalloc (HORIZONTAL_PIXS * MAX_DISPLAY * sizeof(int));
      if (raw_data == NULL) er_abort (S_NO_STORAGE);
   }

   if (reset_display || ! same_buffer) {
      /* Adjust old raw data */
      initial_offset = HORIZONTAL_PIXS - plot_size;
      if (current_size > 0 && plot_size < HORIZONTAL_PIXS) {
         copy_size = (initial_offset > current_size) ? current_size : initial_offset;
         copy_from = HORIZONTAL_PIXS - copy_size;
         copy_to = HORIZONTAL_PIXS - channel_blocksize - copy_size;
         from_ptr = &(raw_data[copy_from]);
         to_ptr = &(raw_data[copy_to]);
         copy_size += HORIZONTAL_PIXS * (current_display->n_channels - 1);
         movedata (FP_SEG(from_ptr), FP_OFF(from_ptr), FP_SEG(to_ptr),
                   FP_OFF(to_ptr), copy_size * sizeof(int));
         current_size += plot_size;
         if (current_size > HORIZONTAL_PIXS) current_size = HORIZONTAL_PIXS;
      } else current_size = plot_size;

      /* Get new raw data */
      index = initial_offset;
      for (i = 0; i < current_display->n_channels; i++) {
         channel_offset = b_head->info.blocksize *
                          ((current_display->type == LIST) ?
                           current_display->channel_list[i] :
                           current_display->begin_channel + i);
         for (; b_head != NULL; b_head = dm_get_next_buffer()) {
            for (k = channel_offset; k < (b_head->info.blocksize + channel_offset);
                 k += h->decimation, index++)
               raw_data[index] = b_head->data[k] - b_head->info.dc_offset;
         }
         index += initial_offset;
         b_head = dm_get_first_buffer();
      }
      reset_display = FALSE;
   }

   if (plot_data == NULL ) {
/*
      plot_data = (int far *) _fmalloc (HORIZONTAL_PIXS *
                              current_display->n_channels * sizeof(int));
*/
      plot_data = (int far *) _fmalloc (HORIZONTAL_PIXS *
                              MAX_DISPLAY * sizeof(int));
      if (plot_data == NULL) er_abort (S_NO_STORAGE);
   }

   /* Compute the scale factor (bit_shift).  Assume that the dynamic range
      remains constant. */
   for (bit_shift = 0; (CHANNEL_HEIGHT >> bit_shift); bit_shift++);
   bit_shift = b_head->dynamic_range - bit_shift;
   bit_shift -= h->magnification;

   /* Generate plot data */
   for (k = 0; k < HORIZONTAL_PIXS * current_display->n_channels; k++) {
      adjust = 0; lsb_mask = 0;
      if (raw_data[k] < 0) {
         adjust = 1;
         lsb_mask = (raw_data[k]+1) & 0x0001;
      }
      plot_data[k] = ((raw_data[k]+lsb_mask) >> bit_shift) + adjust;
   }

   display_traces (current_size);
}

/*=======================================================================*
 *                           s_display_triggers                          *
 *=======================================================================*/
/* Display triggers on the screen.                                       */

PUBLIC
void s_display_triggers ()
{
   unsigned int i, j, k, l, i_increment, channel;
   FLAG clear_flag;

   if (screen_id != SCREEN_1) return;

   i_increment = (int) ((VERTICAL_PIXS) / current_display->orientation);
   i = (TOP_MARGIN) + (i_increment / 2);
   j = LEFT_MARGIN;

   for (k = 0; k < current_display->n_channels; k++) {
      channel = (current_display->type == LIST) ?
                 current_display->channel_list[k] :
                 (current_display->begin_channel + k);
      clear_flag = FALSE;
      switch (t_get_channel_trigger_state(channel)) {
         case CHANNEL_TRIGGERED_EVENT:
            gputs ( i-4, j-8, "E", 1, TRIGGER_ON_COLORS);
            break;
         case CHANNEL_TRIGGERED:
            gputs ( i-4, j-8, "T", 1, TRIGGER_ON_COLORS);
            break;
         case CHANNEL_TRIGGER_ENABLED:
         case CHANNEL_TRIGGER_DISABLED:
         default:
            clear_flag = TRUE;
      }
      switch (dsp_get_status()) {
         case DSP_CALIBRATION:
            if (dsp_get_spectral_channel() ==  channel) {
               gputs ( i-4, j-8, "C", 1, TRIGGER_ON_COLORS);
               clear_flag = FALSE;
            }
            break;
         case OFF:
         default: ;
      }
      if (clear_flag) gputs ( i-4, j-8, " ", 1, TRIGGER_OFF_COLORS);
      i += i_increment;
   }
}

/*=======================================================================*
 *                             s_initialize                              *
 *=======================================================================*/
/* Initialize the screen module.                                         */

PUBLIC
void s_initialize ()
{
   current_display = NULL;
   display_id = -1;
   display[HOME].type = LIST;
   display[HOME].n_channels = 0;
   display[HOME].orientation = 0;
   display[BLOCK].type = RANGE;
   display[BLOCK].n_channels = 0;
   display[BLOCK].orientation = 0;
   display[BLOCK].begin_channel = 0;

   old_size = 0;
   raw_data = plot_data = old_plot_data = NULL;

   /* Header */
   h->magnification = 0;
   h->decimation = 1;
   h->right_time = 0;
}

/*=======================================================================*
 *                          s_initialize_screen                          *
 *=======================================================================*/
/* Initialize the graphics screen.                                       */

PUBLIC
void s_initialize_screen ()
{
   set_graphics();
   turnon( 0);
   screen_id = SCREEN_1;
}

/*=======================================================================*
 *                            s_remove_channel                           *
 *=======================================================================*/
/* Remove a channel from the home display.                               */

PUBLIC
FLAG s_remove_channel (channel)
unsigned int channel;
{
   /* If channel is in home display's channel list,
         remove it and decrement n_channels and orientation
         if screen is on, clear display;
         return (TRUE);
      else return (FALSE);
   */
}

/*=======================================================================*
 *                             s_reset_screen                            *
 *=======================================================================*/
/* Reset the monochrome screen to text.                                  */

PUBLIC
void s_reset_screen ()
{
   screen_id = SCREEN_0;
   set_video (0x3);
#if defined (EGA) || defined (VGA)
#elif defined (MONO)
   set_text ();
#endif
}

/*=======================================================================*
 *                              s_screen_on                              *
 *=======================================================================*/
/* Return screen flag.                                                   */

PUBLIC
FLAG s_screen_on ()
{
   return ((screen_id == SCREEN_0) ? FALSE : TRUE);
}

/*=======================================================================*
 *                            s_select_display                           *
 *=======================================================================*/
/* Select the display mode.                                              */

PUBLIC
void s_select_display (display_mode)
unsigned int display_mode;
{
   screen_id = SCREEN_1;
   clear_display();
   clear_bar();
   display_id = display_mode;
   current_display = &display[display_mode];
}

/*=======================================================================*
 *                            s_select_screen                            *
 *=======================================================================*/
/* Select a screen.                                                      */

PUBLIC
void s_select_screen (screen)
unsigned int screen;
{
   screen_id = screen;
}

/*=======================================================================*
 *                           s_set_channelrange                          *
 *=======================================================================*/
/* Set the channel range in the block display.                           */

PUBLIC
FLAG s_set_channelrange (rel_begin, n_channels)
int rel_begin;
unsigned int n_channels;
{
   unsigned int scan_count;
   unsigned int old_count;

   old_count = display[BLOCK].n_channels;

   scan_count = dt_get_scan_count();
   if (display[BLOCK].begin_channel+rel_begin >= scan_count) return (FALSE);
   if (display[BLOCK].begin_channel+rel_begin < 0) return (FALSE);

   display[BLOCK].begin_channel += rel_begin;
   if ((display[BLOCK].begin_channel+n_channels) >= scan_count)
      display[BLOCK].n_channels = scan_count - display[BLOCK].begin_channel;
   else display[BLOCK].n_channels = n_channels;

   if (display[BLOCK].n_channels > MAX_DISPLAY)
      display[BLOCK].n_channels = MAX_DISPLAY;
   display[BLOCK].orientation = display[BLOCK].n_channels;


   if (screen_id == SCREEN_1) {
      if (old_count != display[BLOCK].n_channels)
         clear_display();
      else {
         current_size = 0;
         reset_display = TRUE;
      }
      display_bar (display[BLOCK].begin_channel, display[BLOCK].n_channels, scan_count);
   }
   return (TRUE);
}

/*=======================================================================*
 *                               s_view_help                             *
 *=======================================================================*/
/* Display the help screen.                                              */

PUBLIC
void s_view_help ()
{
   static int start_i = 50, j = 14;
   int line, i;

   /* Clear the screen */
   turnon( BCKGND_COLOR);

   screen_id = SCREEN_2;

   /* Display lines until NULL is reached */
   for (i = start_i, line = 0; help_info[line]; line++, i += 10)
      gputs (i, j, help_info[line], strlen(help_info[line]), HELP_COLORS);
}
